# go {#go}

## Syntax {#syntax}

`go`

## Details {#details}

In DolphinDB, the entire script submitted for execution is parsed before execution starts. The go statement splits the script into blocks to parse and execute these blocks one by one. The system parses and executes the first block, then parses and executes the second block, and etc.

When parsing, a variable or function must be explicitly defined to be referenced in subsequent code. Some variables or functions are dynamically registered during the execution of functions such as [share](../../Functions/s/share.md), [enableTableShareAndPersistence](../../Functions/e/enableTableShareAndPersistence.md), [loadPlugin](../../Functions/l/loadPlugin.md) and [run](../../Functions/r/run.md), but are not registered during the parsing phase. As a result, if the subsequent code references these dynamically registered variables or functions, we must use the go statement before the subsequent code. Otherwise, exceptions indicating undefined variables or functions will be thrown in parsing subsequent code.

**Note:** A DolphinDB statement or function must be fully parsed to be executed. Therefore, the `go` statement in conditional statements, loops and other nested statements or function bodies will not take effect.

## Examples {#examples}

Example 1. Dynamically register variables

When the following code is executed, the system will throw an exception indicating that the variable is undefined.

``` {#codeblock_egm_bkc_2bc}
t=table(rand(`WMI`PG`TSLA,100) as sym, rand(1..10, 100) as qty, rand(10.25 10.5 10.75, 100) as price)
share(t,`st)
insert into st values(`AAPL,50,10.25);
// Syntax Error: [line #3] Can't recognize table st
```

As the entire script was parsed together before execution and during parsing the `share` statement has not generated the object "st", the insert statement cannot find the object "st". For cases like this, we can use the `go` statement after the share statement to split the script into 2 parts.

``` {#codeblock_lgz_bkc_2bc}
t=table(rand(`WMI`PG`TSLA,100) as sym, rand(1..10, 100) as qty, rand(10.25 10.5 10.75, 100) as price)
share(t,`st)
go;
insert into st values(`AAPL,50,10.25);
```

DolphinDB first parses and executes the first code block before the `go` statement to share table *t* as *st*, then parses and executes the second code block to insert values into *st*.

Example 2. Define variables in script files

In the following example, the variable a=100 is defined in the *test.txt* file, and the file is executed through the `run` function to call `print()` variable a. If the `go` statement is not used, the compilation will report an error that variable a is not defined when variable a is referenced in the subsequent code,

``` {#codeblock_yh5_zjc_2bc}
run("/home/DolphinDB/test.txt");
print(a);
// Syntax Error: [line #2] Cannot recognize the token a
```

Use `go` statement after `run` to define a before `print()`:

``` {#codeblock_vpf_dkc_2bc}
run("/home/DolphinDB/test.txt");
go;
print(a);
```

Example 3. Dynamically register named function

In the following code block, during the parsing phase, the variable fs is registered, but the function body `f2()` is not. Therefore, an error indicating undefined variables would occur as the system goes on to parse the rest of the script.

``` {#codeblock_zh5_zjc_2bc}
fs = ["def f2(){return 'test1';}", "def f2(){return 'test2';}", "def f2(){return 'test3';}"];
runScript(fs[2]);
print(f2());
// Syntax Error: [line #25] Cannot recognize the token f2
```

Use the `go` statement to split the code and let the system parse the first block to generate the dynamic variable `f2()`.

``` {#codeblock_a35_zjc_2bc}
fs = ["def f2(){return 'test1';}", "def f2(){return 'test2';}", "def f2(){return 'test3';}"];
runScript(fs[2]);
go
print(f2());
// output: test3
```

Example 4. `go` in for-loop

The `go` statement doesn't work inside a for-loop statement.

``` {#codeblock_b35_zjc_2bc}
fs = ["def f2(){return 'test1';}", "def f2(){return 'test2';}", "def f2(){return 'test3';}"];
for(s in fs){runScript(s); go; print(f2());}
// Syntax Error: [line #2] Cannot recognize the token f2
```

Example 5. Undefine variables

The following example attempts to define a variable a and then undefine it:

``` {#codeblock_c35_zjc_2bc}
a = [1,2,3,4,5];
b = typestr(a);
b; 
// Execute the above part and b returns FAST INT VECTOR

undef "a";
a = 1;
b = typestr(a);
b; 
// Error: 'a = 1 => Assignment statement failed probably due to invalid indices [a = 1]'
```

Although `undef "a"` intends to remove the original definition of variable a, the variable cannot be dropped until the entire script is parsed. This invalidates the attempt to reassign a on line 7, because the actual value of a in the runtime environment is still the definition of line 1.

A `go` statement must be added after `undef "a"` to split the script into 2 parts.

``` {#codeblock_nvw_hkc_2bc}
a = [1,2,3,4,5];
b = typestr(a);
b;
undef "a";
go;
a = 1;
b = typestr(a);
b;
```

