# Lambda Expression {#lambda-expression}

Lambda expression is a function definition with only one statement.

**Parent topic:**[Functional Programming](../../Programming/FunctionalProgramming/FunctionalProgramming.md)

## Syntax {#syntax}

`def <functionName>(parameters): expression`

or

`def (parameters): expression`

or

`def (parameters) -> expression`

or

`parameter -> expression`

## Details {#details}

Lambda expression can be either a named function or anonymous function. It takes any number of parameters but has only one expression. The second, third, and fourth of the aforementioned syntax define an anonymous Lambda function and the fourth syntax only applies to a statement with one parameter. An anonymous Lambda function can be used in the following ways.

1.  Assigned to a variable, e.g. `f=def(x):x*x`.

2.  Used as the parameter to a function, e.g. `each(def(a,b):a+b, 1..10, 2..11)`.

3.  Used as the return value of a function, e.g. `def(x){return def(k): k*x}`.

4.  Besides the aforementioned usages, Lambda expression can also be wrapped in curly braces \{\}, which supports:

    -   Used as a function with one parameter, e.g. `{x->x*x}(10)`.

    -   Used as an operator. The precedence of binary operators is the same as that of multiplication and division, while unary operators have the lowest precedence. For example, the correct precedence of `{x->x*x} 10 {x,y->x+y} 1` is to first execute `10 {x,y->x+y} 1`, which results in 11, and then apply `{x->x*x}` to 11 and return the final result 121.

    -   Used with symbols of higher-order functions, e.g. `{x->x*x + x}:E matrix(1 2 3, 4 5 6)`.

    -   Called by `object.method()`.

## Examples {#examples}

```
def f(x):x pow 2 + 3*x + 4.0;
f(2);
// output: 14

def orderby(x,y): x[isort y];
t=table(3 1 7 as id, 7 4 9 as value);
orderby(t, t.id);
```

|id|value|
|---|-----|
|1|4|
|3|7|
|7|9|

```
each(def(a,b):a+b, 1..10, 2..11);
// output: [3,5,7,9,11,13,15,17,19,21]

g = def (x,y) -> log(x) + y pow 2;
g(e,0);
// output: 1

each(x->x+1,1 3 5)
// output: [2,4,6]
```

Used as the return value of a function:

``` {#codeblock_ek3_psj_g1c}
def f1(x){return def(k): pow(x,k)}
f1(1)
//output: {def (x, k)->power(x, k)}{1}
f1(1)(10)  
//output: 1
```

Examples of Lambda expression wrapped in curly braces \{\}:

-   Used as a function:

    ``` {#codeblock_y14_qsj_g1c}
    {x->x*x}(2)  
    //output: 4
    
    // Wrap an aggregate function
    {defg(x)-> sum(x)/max(x)}(2 5 6)
    //output: 2
    ```

-   Used as an operator:

    ``` {#codeblock_e1p_rsj_g1c}
    {def(x):x*x} 4
    //output: 16
    ```

    ``` {#codeblock_g5x_rsj_g1c}
    2 *{x->x*x} 10 {x,y->x+y} 1 
    //output: 242
    ```


When used as a unary operator, multiple operators can be used consecutively in a single expression:

``` {#codeblock_kx5_ssj_g1c}
{x->x*x}{x->x+1}{x->x*2} 10
//output: 441
```

Call Lambda expression using `object.method()`.

``` {#codeblock_tbf_4hx_l1c}
v = 1..6 
v.{x->x*x+1}() 
// output [2,5,10,17,26,37]
```

`object.method()` can also be used with higher-order functions. For example, to apply the Lambda expression on each row of a matrix, you can use `object.method()` with `byRow(:H)`:

``` {#codeblock_l2m_phx_l1c}
m = matrix(1 2 3, 4 5 6, 7 8 9) 
m.{x->cumavg(x)*x+x}:H()
```

<table id="table_vzf_nhx_l1c"><tbody><tr><td>

**\#0**

</td><td>

**\#1**

</td><td>

**\#2**

</td></tr><tr><td>

2

</td><td>

14

</td><td>

35

</td></tr><tr><td>

6

</td><td>

22.5

</td><td>

48

</td></tr><tr><td>

12

</td><td>

33

</td><td>

63

</td></tr></tbody>
</table>