# Data Type Conversion {#data-type-conversion}

Data type conversions can be achieved through either data type conversion functions or function [cast\($\)](../../../Functions/c/cast.md).

DolphinDB supports type conversion functions including [string](../../../Functions/s/string.md), [bool](../../../Functions/b/bool.md), [char](../../../Functions/c/char.md), [short](../../../Functions/s/short.md), [int](../../../Functions/i/int.md), [long](../../../Functions/l/long.md), [double](../../../Functions/d/double.md), [date](../../../Functions/d/date.md), [month](../../../Functions/m/month.md), [time](../../../Functions/t/time.md), [second](../../../Functions/s/second.md), [minute](../../../Functions/m/minute.md), [datetime](../../../Functions/d/datetime.md), [timestamp](../../../Functions/t/timestamp.md), [symbol](../../../Functions/s/symbol.md), [nanotime](../../../Functions/n/nanotime.md), [nanotimestamp](../../../Functions/n/nanotimestamp.md), [datehour](../../../Functions/d/datehour.md), [uuid](../../../Functions/u/uuid.md), [ipaddr](../../../Functions/i/ipaddr.md), [int128](../../../Functions/i/int128.md), [blob](../../../Functions/b/blob.md), [complex](../../../Functions/c/complex.md), [point](../../../Functions/p/point.md), [duration](../../../Functions/d/duration.md), [decimal32](../../../Functions/d/decimal32.md), [decimal64](../../../Functions/d/decimal64.md), [decimal128](../../../Functions/d/decimal128.md).

Each individual function has 3 usages:

-   Create a new variable with null value

-   Convert from a string

-   Convert from other data types


**Note:**

-   All these functions \(except for `symbol`\) accept zero or one parameter. If there is no input parameter, it creates a corresponding scalar object with a default value. If the parameter is a string or string vector, it will convert the string to the target data type accordingly. Other types are converted accordingly if they are semantically compatible.

-   The short, int, and long functions use rounding to convert floating-point numbers to integers, and use truncation to convert strings by discarding the fractional part of that given value.


**Parent topic:**[Data Types](../../../Programming/DataTypesandStructures/DataTypes/DataTypes.md)

## string {#string}

```
// make a new string with default value ""
string()=="";
// output: 1

string(10);
// output: 10

typestr string(108.5);
// output: STRING

string(now());
// output: 2016.03.02T20:55:31.287
```

## bool {#bool}

```
x=bool();
x;
// output: 00b
typestr x;
// output: BOOL

bool(`true);
// output
1
bool(`false);
// output: 0

bool(100.2);
// output: 1

bool(0);
// output: 0
```

## decimal32 {#decimal32}

``` {#codeblock_aq2_xym_jfc}
a=decimal32(142, 2)
a
// output: 142.00
b=decimal32(1\7, 6)
b
// output: 0.142857
a+b
// output: 142.142857
a*b
// output: 20.28569400
decimal32("3.1415926535", 4)
// output: 3.1415
```

All elements in a DECIMAL-type vector must have the same data type and scale. For example:

``` {#codeblock_w3w_xym_jfc}
d1=[1.23$DECIMAL32(4), 3$DECIMAL32(4), 3.14$DECIMAL32(4)];
// output: [1.2300,3.0000,3.1400]
typestr(d1);
// output: FAST DECIMAL32 VECTOR
//If the elements have different scales, the sever will create and ouput a tuple.​
d2=[1.23$DECIMAL32(4), 3$DECIMAL32(4), 3.14$DECIMAL32(3)];
// output: (1.2300,3.0000,3.140)
typestr(d2);
// output: ANY VECTOR
```

When converting data of STRING or SYMBOL types to DECIMAL, the behavior differs depending on the server version.

-   For versions earlier than 2.00.10, the extra digits exceeding the scale will be directly ​​truncated​.

    ``` {#codeblock_cgw_yym_jfc}
    symbol(["1.341", "4.5677"])$DECIMAL32(2)
    // output: [1.34,4.56]
    ```

-   For versions 2.00.10 and later, the value will be rounded**​**​ to the nearest valid decimal based on the specified scale.

    ``` {#codeblock_jxf_zym_jfc}
    symbol(["1.341", "4.5677"])$DECIMAL32(2)
    // output: [1.34,4.57]
    ```


## decimal64 {#decimal64}

``` {#codeblock_zjq_zym_jfc}
a=decimal64(142, 2)
a
// output: 142.00
b=decimal64(1\7, 6)
b
// output: 0.142857
a+b
// output: 142.142857
a*b
// output: 20.28569400
decimal64("3.1415926535", 4)
// output: 3.1415
```

All elements in a DECIMAL-type vector must have the same data type and scale. For example:

``` {#codeblock_snl_1zm_jfc}
d1=[1.23$DECIMAL64(4), 3$DECIMAL64(4), 3.14$DECIMAL64(4)];
// output: [1.2300,3.0000,3.1400]
typestr(d1);
// output: FAST DECIMAL64 VECTOR
//If the elements have different scales, the sever will create and ouput a tuple.​
d2=[1.23$DECIMAL64(4), 3$DECIMAL64(4), 3.14$DECIMAL64(3)];
// output: (1.2300,3.0000,3.140)
typestr(d2);
// output: ANY VECTOR
```

When converting data of STRING or SYMBOL types to DECIMAL, the behavior differs depending on the server version.

-   For versions earlier than 2.00.10, the extra digits exceeding the scale will be directly ​​truncated​.

    ``` {#codeblock_uzp_bzm_jfc}
    symbol(["1.341", "4.5677"])$DECIMAL64(2)
    // output: [1.34,4.56]
    ```

-   For versions 2.00.10 and later, the value will be rounded**​**​ to the nearest valid decimal based on the specified scale.

    ``` {#codeblock_dkz_bzm_jfc}
    symbol(["1.341", "4.5677"])$DECIMAL64(2)
    // output: [1.34,4.57]
    ```


## decimal128 {#decimal128}

``` {#codeblock_gll_czm_jfc}
a=decimal128(142, 2)
a
// output: 142.00
b=decimal128(1\7, 6)
b
// output: 0.142857
a+b
// output: 142.142857
a*b
// output: 20.28569400
decimal128("3.1415926535", 4)
// output: 3.1416
```

All elements in a DECIMAL-type vector must have the same data type and scale. For example:

``` {#codeblock_ncd_dzm_jfc}
d1=[1.23$DECIMAL128(4), 3$DECIMAL128(4), 3.14$DECIMAL128(4)]; 
// output: [1.2300,3.0000,3.1400]
typestr(d1);
// output: FAST DECIMAL128 VECTOR
//If the elements have different scales, the sever will create and ouput a tuple.
d2=[1.23$DECIMAL128(4), 3$DECIMAL128(4), 3.14$DECIMAL128(3)];
// output: (1.2300,3.0000,3.140)
typestr(d2);
// output: ANY VECTOR
```

## int {#int}

```
x=int();
x;
// output: 00i

typestr x;
// output: INT

int(`10.9);
// output: 10

int(2147483647);
// output: 2147483647

// maximum value for an INT is 2^31-1=2147483647
int(2147483648);

// output: 00i
```

## short {#short}

```
x=short();
x;
// output:00h

typestr x;
// output: SHORT

short(`12.3);
// output: 12

short(`120.9c);
// output: 120

short(32767);
// output: 32767


/ maximum value for a SHORT is 2^15-1=32767
short(32768);

// output: 00h
```

## long {#long}

```
x=long();
x;
// output: 00l

typestr x;
// output: LONG

long(`10.9);
// output: 10

long(9223372036854775807l);
// output: 9223372036854775807

// maximum value for LONG is 2^63-1=9223372036854775807
long(9223372036854775808l);

// output: 9223372036854775807
```

## char {#char}

```
x=char();
x;
// output: 00c
typestr x;

// output: CHAR

a=char(99);
a;

// output: 'c'

typestr a;

// output: CHAR

char(a+5);

// output: 'h'
```

## double {#double}

```
x=double();
x;

// output: 00F

typestr x;

// output: DOUBLE

typestr double(`10);

// output: DOUBLE

double(`10.9);

// output: 10.9

double(now());

// output: 5.297834e+011
```

## date {#date}

```
date();

// output: 00d

date(`2011.10.12);

// output: 2011.10.12

date(now());

// output: 2016.03.02
```

## datehour {#datehour}

``` {#codeblock_chp_fzm_jfc}
datehour(1)
// output: 1970.01.01T01
datehour(2012.06.13 13:30:10);
// output: 2012.06.13T13
datehour([2012.06.15 15:32:10.158,2012.06.15 17:30:10.008]);
// output: [2012.06.15T15,2012.06.15T17]
datehour(2012.01M)
// output: 2012.01.01T00
```

## datetime {#datetime}

```
datetime(2009.11.10);

// output: 2009.11.10T00:00:00

typestr datetime(2009.11.10);

// output: DATETIME

datetime(now());

// output: 2016.03.02T20:51:10
```

## timestamp {#timestamp}

```
timestamp(2016.10.12);

// output: 2016.10.12T00:00:00.000

timestamp(2016.10.12)+1;

// output: 2016.10.12T00:00:00.001

timestamp(now());

// output: 2016.10.13T20:28:45.104
```

## month {#month}

```

 month();

// output: 0M


 month(`2012.12m);


// output: 012.12M


/make a month variable from date

 month(2012.12.23);


// output: 012.12M

//make a month variable from timestamp

 month(now());


// output: 016.03M
```

## second {#second}

```
second();

// output: 00s

second("19:36:12");

// output: 19:36:12

second(now());

// output: 20:50:31
```

## minute {#minute}

```
minute();

// output: 00m

minute(now());

// output: 20:49m
```

## hour {#hour}

``` {#codeblock_qvt_hzm_jfc}
hour(2012.12.03 01:22:01);
// output: 1
```

## time {#time}

```
time();

// output: 00t

time("12:32:56.356");

// output: 12:32:56.356

time(now());

// output: 20:49:12.564
```

## symbol {#symbol}

```
x=`AMZN`AAPL`GOOG`FB`SNAP;

x;

// output: ["AMZN","AAPL","GOOG","FB","SNAP"]

typestr x;

// output: STRING VECTOR

y=symbol(x);

y;

// output: ["AMZN","AAPL","GOOG","FB","SNAP"]

typestr y;

// output: FAST SYMBOL VECTOR
```

## nanotime {#nanotime}

``` {#codeblock_yb1_kzm_jfc}
nanotime(1000000000);
// output: 00:00:01.000000000
nanotime(12:06:09 13:08:01);
// output: [12:06:09.000000000,13:08:01.000000000]
nanotime(2012.12.03 01:22:01.123456789);
// output: 01:22:01.123456789
nanotime('13:30:10.008007006');
// output: 13:30:10.008007006
```

## nanotimestamp {#nanotimestamp}

``` {#codeblock_skh_kzm_jfc}
nanotimestamp(1);
// output: 1970.01.01 00:00:00.000000001
nanotimestamp(1000000000);
// output: 1970.01.01 00:00:01.000000000
nanotimestamp(2012.12.03 12:06:09 2012.12.03 13:08:01);
// output: [2012.12.03 12:06:09.000000000,2012.12.03 13:08:01.000000000]
nanotimestamp(2012.12.03 01:22:01.123456789);
// output: 2012.12.03 01:22:01.123456789
nanotimestamp('2012.12.03 13:30:10.008007006');
// output: 2012.12.03 13:30:10.008007006
nanotimestamp(now());
// output: 2024.02.22 16:14:28.627000000
nanotimestamp(2012.01M)
// output: 2012.01.01T00:00:00.000000000

```

## uuid {#uuid}

``` {#codeblock_iy3_nzm_jfc}
uuid("");
// output: 00000000-0000-0000-0000-000000000000
a=uuid("9d457e79-1bed-d6c2-3612-b0d31c1881f6");
a;
// output: 9d457e79-1bed-d6c2-3612-b0d31c1881f6
typestr(a);
// output: UUID
```

## ipaddr {#ipaddr}

``` {#codeblock_ukw_nzm_jfc}
a=ipaddr("192.168.1.13");
a;
// output: 192.168.1.13
typestr(a);
// output: IPADDR
```

## int128 {#int128}

``` {#codeblock_w22_4zm_jfc}
a=int128("e1671797c52e15f763380b45e841ec32")
// output: e1671797c52e15f763380b45e841ec32
typestr(a);
// output: INT128
```

## duration {#duration}

``` {#codeblock_xr4_4zm_jfc}
y=duration("20H")
y
// output: 20H
typestr(y)
// output: DURATION
duration("3XNYS")
// output: 3XNYS
```

When the unit of time is not specified, the system defaults it as seconds \(s\):

``` {#codeblock_wrd_pzm_jfc}
t=table(take(2018.01.01T01:00:00+1..10,10) join take(2018.01.01T02:00:00+1..10,10) join take(2018.01.01T08:00:00+1..10,10) as time, rand(1.0, 30) as x);
select max(x) from t group by bar(time, 5);
```

|bar\_time|max\_x|
|---------|------|
|2018.01.01T01:00:00|0.8824|
|2018.01.01T01:00:05|0.8027|
|2018.01.01T01:00:10|0.572|
|2018.01.01T02:00:00|0.8875|
|2018.01.01T02:00:05|0.8542|
|2018.01.01T02:00:10|0.4287|
|2018.01.01T08:00:00|0.9294|
|2018.01.01T08:00:05|0.9804|
|2018.01.01T08:00:10|0.2147|

Specify the unit of time as minute \(m\) and group time column with 1-minute intervals:

``` {#codeblock_m5n_qzm_jfc}
select max(x) from t group by bar(time, 1m);
```

|bar\_time|max\_x|
|---------|------|
|2018.01.01T01:00:00|0.8824|
|2018.01.01T02:00:00|0.8875|
|2018.01.01T08:00:00|0.9804|

## cast\(X, dataTypeName\) / $ {#castx-datatypename--}

```
x=3.1;
typestr x;

// output: DOUBLE

x=cast(x, int);
// can also use cast(x, INT)

x;
// output: 3
typestr x;
// output: INT

19.99$INT;
// output: 20

syms =`IBM`C`MS`MSFT`JPM`ORCL`BIDU`SOHU
typestr syms;
// output: STRING VECTOR
syms=syms$SYMBOL;
typestr syms;
// output: FAST SYMBOL VECTOR
```

For more about the `cast` function please see [cast](../../../Functions/c/cast.md).

