# schema {#schema}

## Syntax {#syntax}

`schema(table|dbHandle)`

## Arguments {#arguments}

**table \| dbHandle** can be a table object or a database handle.

## Details {#details}

Display information about the schema of a table or a database. The function returns an unordered dictionary containing the following information \(in alphabetical order\):

-   atomic: the level at which the atomicity is guaranteed for a write transaction. It can be TRANS or CHUNK.
-   chunkGranularity: the chunk granularity which determines whether to allow concurrent writes to different tables in the same chunk. It can be TABLE or DATABASE.
-   clusterReplicationEnabled: whether asynchronous replication has been enabled.
-   colDefs: information about each column in the table:
    -   name: column name
    -   typeString: column type
    -   typeInt: the type ID
    -   extra: the scale of a DECIMAL value
    -   comment: comment on the column
    -   sensitive: whether this column is set as a sensitive column. It is returned only when table is a DFS table.
-   compressMethods: the compression methods used for specified columns of a DFS table.
    -   name: Column names
    -   compressMethods: Compression algorithm applied, including "lz4", "delta", or "zstd".
-   databaseDir: the directory where the database is stored.
-   databaseOwner: the database creator.
-   dbUrl: path to the distributed database where the DFS table is located. It is returned only when *table* is a DFS table.
-   encryptMode: The table encryption mode specified during table creation. It is returned only when *table* is a DFS table.
-   engineType: the storage engine type. It can be OLAP or TSDB.
-   keepDuplicates: how to deal with records with duplicate sort key values.
-   keyColumn: key column\(s\) of the table. It is returned only when *table* contains key column\(s\).
-   partitionColumnIndex: the index that indicates the positions of partitioning columns. It returns -1 for a dimension table.
-   partitionColumnName: the partitioning column names.
-   partitionColumnType: the data type ID \(which can be checked at Data Types and Data Forms\) of the partitioning column.
-   partitionSchema: how the partitions are organized
-   partitionSites \(optional\): If the parameter locations is specified for function `database`, it displays the ip:port information of the specified node.
-   partitionTypeName / partitionType: the partitioning scheme and the corresponding ID, including VALUE\(1\), RANGE\(2\), LIST\(3\), COMPO\(4\), HASH\(5\).
-   sortColumns: the sort columns for a table.
-   softDelete: whether soft deletion of the table has been enabled. This field will only be returned for tables in TSDB databases. Return true if soft deletion has been enabled, false otherwise.
-   sortKeyMappingFunction: the mapping functions applied to sort keys.
-   tableComment: the comment for DFS tables.
-   tableName: table name. It is returned only when *table* is a DFS table.
-   tableOwner: the table creator.
-   partitionFunction: the function applied to a column for data partitioning. It is a STRING vector with each element as a function signature. If an element is "asis", meaning no function is applied to that column. For example, `partitionFunction->["myPartitionFunc{, 6, 8}","asis"]`.
-   latestKeyCache: whether the latest value cache is enabled.
-   compressHashSortKey: whether the compression for sort columns is enabled.

## Examples {#examples}

OLAP database:

```
n=1000000
ID=rand(10, n)
x=rand(1.0, n)
t=table(ID, x)
db=database("dfs://rangedb101", RANGE, 0 5 10)
pt = db.createPartitionedTable(t, `pt, `ID)
pt.append!(t)
pt=loadTable(db,`pt);

schema(db);
/* output:
databaseDir->dfs://rangedb101
partitionSchema->[0,5,10]
partitionSites->
atomic->TRANS
chunkGranularity->TABLE
partitionType->2
partitionTypeName->RANGE
partitionColumnType->4
clusterReplicationEnabled->1
databaseOwner->admin
*/
```

OLAP table:

```
schema(pt);
/* output:
chunkGranularity->TABLE
tableOwner->admin
compressMethods->
name compressMethods
---- ---------------
ID   lz4            
x    lz4            

colDefs->
name typeString typeInt comment
---- ---------- ------- -------
ID   INT        4              
x    DOUBLE     16             

chunkPath->
partitionColumnIndex->0
partitionColumnName->ID
partitionColumnType->4
partitionType->2
partitionTypeName->RANGE
partitionSchema->[0,5,10]
partitionSites->
*/
```

TSDB database:

``` {#codeblock_kds_rzs_c1c}
n = 10000
SecurityID = rand(`st0001`st0002`st0003`st0004`st0005, n)
sym = rand(`A`B, n)
TradeDate = 2022.01.01 + rand(100,n)
TotalVolumeTrade = rand(1000..3000, n)
TotalValueTrade = rand(100.0, n)
schemaTable_snap = table(SecurityID, TradeDate, TotalVolumeTrade, TotalValueTrade).sortBy!(`SecurityID`TradeDate)

dbPath = "dfs://TSDB_STOCK"
if (existsDatabase(dbPath)){dropDatabase(dbPath)}
db_snap = database(dbPath, VALUE, 2022.01.01..2022.01.05, engine='TSDB')

schema(db_snap)
/* output:
databaseDir->dfs://TSDB_STOCK
partitionSchema->[2022.01.01,2022.01.02,2022.01.03,2022.01.04,2022.01.05]
partitionSites->
engineType->TSDB
atomic->TRANS
chunkGranularity->TABLE
partitionType->1
partitionTypeName->VALUE
partitionColumnType->6
clusterReplicationEnabled->1
databaseOwner->admin
*/
```

TSDB table:

``` {#codeblock_wdy_szs_c1c}
def myHashFunc(x){
  return hashBucket(x, 10)
}
snap=createPartitionedTable(dbHandle=db_snap, table=schemaTable_snap, tableName="snap", partitionColumns=`TradeDate, sortColumns=`SecurityID`TradeDate, keepDuplicates=ALL, sortKeyMappingFunction=[myHashFunc])

schema(snap)
/* output:
engineType->TSDB
keepDuplicates->ALL
chunkGranularity->TABLE
sortColumns->["SecurityID","TradeDate"]
sortKeyMappingFunction->["def myHashFunc(x){
  return hashBucket(x, 10)
}"]
softDelete->0
tableOwner->admin
compressMethods->
name             compressMethods
---------------- ---------------
SecurityID       lz4            
TradeDate        lz4            
TotalVolumeTrade lz4            
TotalValueTrade  lz4            

colDefs->
name             typeString typeInt extra comment
---------------- ---------- ------- ----- -------
SecurityID       SYMBOL     17                   
TradeDate        DATE       6                    
TotalVolumeTrade INT        4                    
TotalValueTrade  DOUBLE     16                   

chunkPath->
partitionColumnIndex->1
partitionColumnName->TradeDate
partitionColumnType->6
partitionType->1
partitionTypeName->VALUE
partitionSchema->[2022.01.01,2022.01.02,2022.01.03,2022.01.04,2022.01.05]
partitionSites->
*/
```

In the following example, we use a composite strategy to partition data by “id” and “ts”, creating a measurement point table named “pt”. This table uses “ticket” and “id2” as the columns that uniquely identify a measurement point. The latest value cache is enabled and the column “value” is of IOTANY type, allowing it to store data of various types. The compression for the sort key columns is also enabled.

``` {#codeblock_gfs_lhb_3gc}
dbName = "pt"
if(existsDatabase(dbName)){
        dropDatabase(dbName)
}
db1 = database(, partitionType=HASH, partitionScheme=[INT, 10])
db2 = database(, partitionType=VALUE, partitionScheme=2017.08.07..2017.08.11)
db = database(dbName, COMPO, [db1, db2], engine='IOTDB')
create table "dfs://db"."pt" (
        id INT,
        ticket SYMBOL,
        id2 LONG,
        ts TIMESTAMP,
        id3 IOTANY
)
partitioned by id, ts,
sortColumns = [`ticket, `id2, `ts],
sortKeyMappingFunction = [hashBucket{, 50}, hashBucket{, 50}],
latestKeyCache = true,
compressHashSortKey = true
print schema(loadTable(dbName, `pt))
```

The output:

``` {#codeblock_w53_mhb_3gc}
/* 
engineType->IOTDB
keepDuplicates->ALL
chunkGranularity->TABLE
sortColumns->["deviceId","location","timestamp"]
sortKeyMappingFunction->["hashBucket{, 50}","hashBucket{, 50}"]
softDelete->0
tableOwner->admin
compressMethods->name      compressMethods
--------- ---------------
deviceId  lz4            
location  lz4            
timestamp lz4            
value     lz4            
tableComment->
latestKeyCache->1
compressHashSortKey->0
colDefs->name      typeString typeInt extra comment
--------- ---------- ------- ----- -------
deviceId  INT        4                    
location  SYMBOL     17                   
timestamp TIMESTAMP  12                   
value     IOTANY     41                   
chunkPath->
partitionColumnIndex->[0,2]
partitionColumnName->["deviceId","timestamp"]
partitionColumnType->[4,6]
partitionType->[5,1]
partitionTypeName->["HASH","VALUE"]
partitionSchema->(20,[2017.08.07,2017.08.08,2017.08.09,2017.08.10,2017.08.11,2024.10.12])
partitionSites->
*/
```

