# rowAlign {#rowalign}

## Syntax {#section_v14_1hf_czb .section}

`rowAlign(left, right, how)`

## Arguments {#section_cqc_bhf_czb .section}

**left**/**right** is an array vector or a columnar tuple.

-   *left* and *right* must be of the same data type and size \(number of rows\), but the number of elements in corresponding rows do not have to match. For example, if *left* has 3 rows and the first row has 5 elements, then *right* must also have 3 rows but the first row does not necessarily have 5 elements.

-   Data in each row of *left* and *right* must be strictly increasing/decreasing.


**how** is a string indicating how *left* and *right* will be aligned. It can take the following values:

|*how* \(case-insensitive\)|Description|Max. Value in the Alignment Result|Min. Value in the Alignment Result|
|--------------------------|-----------|----------------------------------|----------------------------------|
|"bid"|*left* and *right* are bid prices sorted in strictly decreasing order. The output will only include the indices of prices that fall within the alignment range.|max\(max\(left\), max\(right\)\)|max\(min\(left\), min\(right\)\)|
|"allBid"|*left* and *right* are bid prices sorted in strictly decreasing order. The output will include the indices of all prices from *left*and *right*.|max\(max\(left\), max\(right\)\)|min\(min\(left\), min\(right\)\)|
|"ask"|*left* and *right* are ask prices sorted in strictly increasing order. The output will only include the indices of prices that fall within the alignment range.|min\(max\(left\), max\(right\)\)|min\(min\(left\), min\(right\)\)|
|"allAsk"|*left* and *right* are ask prices sorted in strictly increasing order. The output will include the indices of all prices from *left*and *right*.|max\(max\(left\), max\(right\)\)|min\(min\(left\), min\(right\)\)|

## Details {#section_scr_2hf_czb .section}

The `rowAlign` function aligns corresponding rows from *left* and *right*based on their values. It returns a tuple containing two elements \(either array vectors or columnar tuples, matching the input type\). Each element in this tuple contains indices that map the aligned elements to their original positions in *left*and *right* respectively. Unmatched elements from one input are marked as -1 in the other input's returned index.

This function is typically used for aligning bid/ask prices, where *left* represents prices from one time point and *right* represents prices from a previous time point. The returned indices can be used with the `rowAt` function to extract the aligned elements from the original *left* and *right* arrays, with any unaligned elements left blank.

The alignment process for each row is illustrated in the accompanying images, where blue blocks indicate the elements not included in the output when using the "bid" or "ask" method.

-   *how* = "bid" or "allBid"

    ![](../../images/rowAlign_1.png)

-   *how* = "ask" or "allAsk":

    ![](../../images/rowAlign_2.png)


## Examples {#section_d55_khf_czb .section}

``` {#codeblock_zkl_lhf_czb}
left = array(DOUBLE[], 0, 5).append!([9.01 9.00 8.99 8.98 8.97, 9.00 8.98 8.97 8.96 8.95, 8.99 8.97 8.95 8.93 8.91])
right = array(DOUBLE[], 0, 5).append!([9.02 9.01 9.00 8.99 8.98, 9.01 9.00 8.99 8.98 8.97, 9.00 8.98 8.97 8.96 8.95])
leftIndex, rightIndex = rowAlign(left, right, "bid")
leftIndex
// output: [[-1,0,1,2,3],[-1,0,-1,1,2],[-1,0,-1,1,-1,2]]

left.rowAt(leftIndex)
// output: [[,9.01,9.00,8.99,8.98],[,9,,8.99,8.97],[,8.99,,8.97,,8.95]]

rightIndex
// output: [[0,1,2,3,4],[0,1,2,3,4],[0,-1,1,2,3,4]]

right.rowAt(rightIndex)
// output: [[9.02,9.01,9.00,8.99,8.98],[9.01,9.00,8.99,8.98,8.97],[9.00,,8.98,8.97,8.96,8.95]]

// output all bid prices in one array vector after aligning left and right
left.rowAt(leftIndex).nullFill(right.rowAt(rightIndex))
// output: [[9.02,9.01,9,8.99,8.98],[9.01,9.00,8.99,8.98,8.97],[9.00,8.99,8.98,8.97,8.96,8.95]]

// the bid sizes
leftBidQty = array(INT[], 0, 5).append!([10 5 15 20 13, 12 15 20 21 18, 7 8 9 9 10])
rightBidQty = array(INT[], 0, 5).append!([8 12 10 12 8, 10 5 15 18 13, 12 15 20 21 19])

// calculate the difference in bid quantities between left and right 
leftBidQty.rowAt(leftIndex).nullFill(0) - rightBidQty.rowAt(rightIndex).nullFill(0)
// output: [[-8,-2,-5,3,12],[-10,7,-15,-3,7],[-12,7,-15,-12,-21,-10]]

leftIndex, rightIndex = rowAlign(left, right, "allBid")
leftIndex
// output: [[-1,0,1,2],[-1,-1,0,1,2],[-1,0,-1,1,2]]

rightIndex
// output: [[0,1,2,-1],[0,1,2,-1,-1],[0,-1,1,2,-1]]
```

``` {#codeblock_yjt_lhf_czb}
left = array(DOUBLE[], 0, 3).append!([8.99 9.00 9.01, 8.97 8.99 9.00, 8.95 8.97 8.99])
right = array(DOUBLE[], 0, 3).append!([9.00 9.01 9.02, 8.99 9.00 9.01, 8.97 8.98 9.00])
leftIndex, rightIndex = rowAlign(left, right, "ask")
leftIndex
// output: [[0,1,2],[0,1,2],[0,1,-1,2]]
 
rightIndex
// output: [[-1,0,1],[-1,0,1],[-1,0,1,-1]]

leftIndex, rightIndex = rowAlign(left, right, "allAsk")
leftIndex
// output: [[0,1,2,-1],[0,1,2,-1],[0,1,-1,2,-1]]
 
rightIndex
// output: [[-1,0,1,2],[-1,0,1,2],[-1,0,1,-1,2]]
```

``` {#codeblock_cxr_hdr_zbc}
sym = `st1`st2`st3
left = [[3.1,2.5,2.8], [3.1,3.3], [3.2,2.9,3.3]]
left.setColumnarTuple!()
right = [[3.1,2.5,2.8], [3.1,3.3], [3.2,2.9,3.3]]
right.setColumnarTuple!()
rowAlign(left, right, "bid")
// output: [([0,1,2],[0,1],[0,1,2]), ([0,1,2],[0,1],[0,1,2])]
```

**Parent topic:**[Functions](../../Functions/category.md)

