# differentialEvolution {#differentialEvolution}

**Parent topic:**[Functions](../../Functions/category.md)

## Syntax {#syntax}

`differentialEvolution(func, bounds, [X0], [maxIter=1000], [popSize=15], [mutation], [recombination=0.7], [tol=0.01], [atol=0], [polish=true], [seed])`

## Arguments {#arguments}

**func** is the objective function to be minimized. Note that the function must return a scalar.

**bounds** is a numeric matrix of shape \(N, 2\) indicating the bounds for parameters, where N is the number of parameters to be optimized.

**X0** \(optional\) is a numeric vector indicating the initial guess to the minimization.

**Note**:

-   Each row in the *bound* parameter contains two values \(min, max\), which define the lower and upper limits for the parameter values specified by *X0*.
-   *X0* and *bounds* must have the same length, i.e., N = size\(*X0*\).

**maxIter** \(optional\) is a non-negative integer indicating the maximum number of iterations. The default value is 1000.

**popSize** \(optional\) is a positive integer specifying the multiplier for setting the total population size. The population contains `popSize*(N - N_equal)` individuals, where `N_equal` represents the number of parameters whose bounds are equal. The default value is 15.

**mutation** \(optional\) is a numeric pair in the format of \(min, max\), indicating the range of the mutation constant. It should satisfy 0 &lt;= min &lt;= max &lt; 2. The default value is \(0.5, 1\).

**recombination** \(optional\) is a numeric scalar in \[0, 1\], indicating the recombination constant, also known as the crossover probability.

**tol** \(optional\) is a non-negative floating-point scalar indicating the relative tolerance for convergence. The default value is 0.01.

**atol** \(optional\) is a non-negative floating-point scalar indicating the absolute tolerance for convergence. The default value is 0. The algorithm terminates when `stdev(population_energies) <= atol + tol * abs(mean(population_energies))`, where `population_energies` is the vector consisting of objective function values for all individuals in the population.

**polish** \(optional\) is a Boolean scalar indicating whether to polish the differential evolution result using the L-BFGS-B method. The default value is true.

**seed** \(optional\) is an integer indicating the random seed used in the differential evolution algorithm, allowing users to reproduce the results. If unspecified \(default\), a non-deterministic random number generator is used.

## Details {#details}

Use the Differential Evolution algorithm to calculate the global minimum of a function with multiple variables.

**Return value**: A dictionary containing the following keys:

-   xopt: A floating-point vector indicating the parameter values that minimize the objective function.
-   fopt: A floating-point scalar indicating the minimum value of the objective function, where fopt = f\(xopt\).
-   iterations: An integer indicating the number of iterations during the optimization process.
-   fcalls: An integer indicating the number of times the objective function is called during the optimization process.
-   converged: A Boolean scalar indicating whether the optimization result is converged.
    -   true: The optimization result has been converged to below a preset tolerance and the algorithm terminates.
    -   false: The algorithm terminates without converging after reaching the maximum number of iterations.

## Examples {#examples}

The following example creates a user-defined function `rosen` and uses `differentialEvolution` \(with *bounds* specified\) to calculate the global minimum of `rosen`.

``` {#codeblock_gc5_z2s_cdc}
def rosen(x) { 
	N = size(x);
	return sum(100.0*power(x[1:]-power(x[:N-1], 2.0), 2.0)+power(1-x[:N-1], 2.0));
}
bounds = matrix([0 0 0 0 0, 2 2 2 2 2])
differentialEvolution(rosen, bounds)

/* Ouput:
fcalls->43656
xopt->[1.000000000000,1.000000000000,1.000000000000,1.000000000000,1.000000000000]
fopt->0.0
iterations->581
converged->true
*/
```

