# cubicHermiteSplineFit {#cubichermitesplinefit}

**Parent topic:**[Functions](../../Functions/category.md)

## Syntax {#syntax}

`cubicHermiteSplineFit(X, Y, derivative, [extrapolate=true])`

## Arguments {#arguments}

**X** is a numeric vector representing the x-coordinates \(independent variable\) of the interpolation points. *X* must be strictly increasing and contain at least two elements.

**Y** is a numeric vector of the same length as *X*, representing the y-coordinates \(dependent variable\) of the interpolation points.

**derivative** is a numeric vector of the same length as *X*, representing the first derivatives of *Y* with respect to *X*.

**extrapolate** \(optional\) is a Boolean scalar indicating whether to allow extrapolation when prediction points fall outside the data range. Defaults to true.

## Details {#details}

This function performs cubic Hermite interpolation on the given vectors *X* and *Y*.

**Return value:** A dictionary with the following keys:

-   modelName: A string "CubicHermiteSpline" indicating the model name.
-   X: The numeric vector representing the x-coordinates used for interpolation \(i.e., the input *X*\).
-   extrapolate: A Boolean scalar. The input *extrapolate*.
-   coeffs: A numeric vector containing the polynomial coefficients fitted from the input data points.
-   predict: The prediction function of the model. You can call `model.predict(X)` or `predict(model, X)` to make predictions with the generated model.
    -   model: The output dictionary of `cubicHermiteSplineFit`.
    -   X: A numeric vector representing the x-coordinates.

## Examples {#examples}

Perform cubic Hermite interpolation on the given vectors *x* and *y*.

``` {#codeblock_d5m_dyj_b2c}
x = [0.0000,0.0800,0.1000,0.1700,0.2000,0.2500,0.3000,0.4000,0.5000,0.6000,0.7000,0.7500,0.8000,0.9000,1.0000]
y = [1.2157,1.4000,1.4000,1.4000,1.4055,1.4323,1.4585,1.5096,1.5291,1.4985,1.4562,1.4440,1.4427,1.4461,1.4504]
N = shape(x)[0]
derv = take([0.0], N)
derv[0] = (y[1] - y[0]) \ (x[1] - x[0])
derv[N-1]  = (y[N-1] - y[N-2]) \ (x[N-1] - x[N-2])
for (i in 1..(N-2)) {
	derv[i] = (y[i+1] - y[i-1]) \ (x[i+1] - x[i-1])
}

model = cubicHermiteSplineFit(x, y, derv)
model;

/* output:
modelName->CubicHermiteSpline
X->[0,0.08,0.1,0.17,0.2,0.25,0.3,0.4,0.5,0.6,0.7,0.75,0.8,0.9,1]
coeffs->#0                   #1                   #2                 #3    
-------------------- -------------------- ------------------ ------
-71.992187499999985  5.759375             2.303749999999999  1.2157
4607.499999999995452 -184.299999999999869 1.842999999999999  1.4   
11.224489795918488   -0.785714285714294   0                  1.4   
102.314814814811015  1.208333333333494    0.055000000000001  1.4   
-55.2999999999993    5.409999999999942    0.40375            1.4055
-1.066666666666638   -0.066666666666655   0.53               1.4323
-15.366666666666818  1.493333333333353    0.515333333333334  1.4585
-9.249999999999774   -0.655000000000037   0.353              1.5096
19.199999999999896   -4.424999999999982   -0.055500000000001 1.5291
11.816666666666725   -1.766666666666675   -0.3645            1.4985
-4.133333333332982   2.593333333333321    -0.363333333333333 1.4562
-27.600000000001628  3.560000000000105    -0.134999999999998 1.444 
-1.549999999999794   0.354999999999965    0.014              1.4427
-0.450000000000062   0.090000000000012    0.038499999999999  1.4461

extrapolate->1
predict->cubicHermiteSplinePredict
*/
```

Evaluate the interpolation model at the given x points with `predict`.

``` {#codeblock_grf_2yj_b2c}
preds = predict(model, x)
preds;

// output: [1.2157,1.4,1.4,1.4,1.4055,1.4323,1.4585,1.5096,1.5291,1.4985,1.4562,1.444,1.4427,1.4461,1.4504]
```

