# bondYieldCurveBuilder {#bondYieldCurveBuilder}

**Parent topic:**[Functions](../../Functions/category.md)

## Syntax {#syntax}

`bondYieldCurveBuilder(referenceDate, currency, bonds, terms, quotes, dayCountConvention, [compounding='Compounded'], [frequency='Annual'], [curveName], [method='Bootstrap'], [interpMethod='Linear'], [extrapMethod='Flat'])`

## Details {#topic_vwq_fyq_ngc}

Builds a bond yield curve to be used as a discount curve \(i.e., a spot yield curve\) in pricing applications.

**Return value**: A MKTDATA object.

## Arguments {#arguments}

**referenceDate** A DATE scalar, representing the reference date of the yield curve.

**currency** A STRING scalar specifying the currency code of the curve. Supported values: "CNY", "USD", "EUR", "GBP", "JPY", "HKD".

**bonds** An INSTRUMENT vector, specifying the sample bonds used for curve construction with bond metadata. See [Bond Product Field Specifications](#topic_vw4_kyq_ngc) for details.

**terms** A DURATION vector of the same length as *bonds*, indicating the remaining term to maturity of each bond.

**quotes** A numeric vector of the same length as *bonds*, indicating the yield to maturity \(YTM\) of each bond on the *referenceDate*.

**dayCountConvention**A STRING scalar indicating the day count convention to use. It can be:

-   "Actual360": actual/360

-   "Actual365": actual/365

-   "ActualActualISDA" —actual/actual according to ISDA \(International Swaps and Derivatives Association\) convention

-   "ActualActualISMA" — actual/actual according to ISMA \(International Securities Market Association\) convention


**compounding** \(optional\) A STRING scalar defining the compounding method. Options:

-   "Compounded" \(default\): Discrete compounding

-   "Simple": Simple interest

-   "Continuous": Continuous compounding


**frequency** \(optional\) A STRING scalar specifying the interest payment frequency. It does not affect curve construction and is only recorded in the returned curve. Supported values:

-   "Annual" \(default\): Annually

-   "NoFrequency": No payment frequency

-   "Once": Single payment at maturity \(for discount bonds and zero coupon bonds only\)

-   "Semiannual": Semiannually

-   "EveryFourthMonth": Every four months

-   "Quarterly": Quarterly

-   "BiMonthly": Every two months

-   "Monthly": Monthly

-   "EveryFourthWeek": Every four weeks

-   "BiWeekly": Every two weeks

-   "Weekly": Weekly

-   "Daily": Daily

-   "Other": Other frequencies


**curveName** \(optional\) A STRING scalar indicating the yield curve name. The default value is NULL.

**method** \(optional\) A STRING scalar specifying the curve construction method. Options:

-   "Bootstrap" \(default\): Bootstrap method

-   "NS": Nelson-Siegel model

-   "NSS": Nelson-Siegel-Svensson model


**interpMethod** \(optional\) A STRING scalar specifying the interpolation method. Options:

-   "Linear" \(default\): Linear interpolation

-   "CubicSpline": Cubic spline

-   "CubicHermiteSpline": Cubic Hermite spline


**extrapMethod** \(optional\) A STRING scalar for the extrapolation method. Options:

-   "Flat" \(default\): Flat extrapolation

-   "Linear": Linear extrapolation


## Examples {#examples}

Build a yield curve based on the Chinese government bond market data as of August 18, 2025.

``` {#codeblock_y55_32x_mgc}
bond1 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "DiscountBond",
    "version": 0, 
    "instrumentId": "259916.IB",
    "start": 2025.03.13,
    "maturity": 2025.09.11,
    "issuePrice":  99.2070,
    "dayCountConvention": "ActualActualISDA"
}
bond2 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "FixedRateBond",
    "version": 0, 
    "instrumentId": "240021.IB",
    "start": 2024.10.25,
    "maturity": 2025.10.25,
    "issuePrice": 100,
    "coupon": 0.0133,
    "frequency": "Annual",
    "dayCountConvention": "ActualActualISDA"
}
bond3 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "FixedRateBond",
    "version": 0, 
    "instrumentId": "250001.IB",
    "start": 2025.01.15,
    "maturity": 2026.01.15,
    "issuePrice": 100,
    "coupon": 0.0116,
    "frequency": "Annual",
    "dayCountConvention": "ActualActualISDA"
}
bond4 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "FixedRateBond",
    "version": 0, 
    "instrumentId": "250013.IB",
    "start": 2025.07.25,
    "maturity": 2026.07.25,
    "issuePrice": 100,
    "coupon": 0.0133,
    "frequency": "Annual",
    "dayCountConvention": "ActualActualISDA"
}
bond5 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "FixedRateBond",
    "version": 0, 
    "instrumentId": "250012.IB",
    "start": 2025.06.15,
    "maturity": 2027.06.15,
    "issuePrice": 100,
    "coupon": 0.0138,
    "frequency": "Annual",
    "dayCountConvention": "ActualActualISDA"
}
bond6 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "FixedRateBond",
    "version": 0, 
    "instrumentId": "250010.IB",
    "start": 2025.05.25,
    "maturity": 2028.05.25,
    "issuePrice": 100,
    "coupon": 0.0146,
    "frequency": "Annual",
    "dayCountConvention": "ActualActualISDA"
}
bond7 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "FixedRateBond",
    "version": 0, 
    "instrumentId": "250014.IB",
    "start": 2025.07.25,
    "maturity": 2030.07.25,
    "issuePrice": 100,
    "coupon": 0.0155,
    "frequency": "Annual",
    "dayCountConvention": "ActualActualISDA"
}
bond8 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "FixedRateBond",
    "version": 0, 
    "instrumentId": "2500802.IB",
    "start": 2025.05.25,
    "maturity": 2032.05.25,
    "issuePrice": 100,
    "coupon": 0.0157,
    "frequency": "Annual",
    "dayCountConvention": "ActualActualISDA"
}
bond9 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "FixedRateBond",
    "version": 0, 
    "instrumentId": "250011.IB",
    "start": 2025.05.25,
    "maturity": 2035.05.25,
    "issuePrice": 100,
    "coupon": 0.0167,
    "frequency": "Semiannual",
    "dayCountConvention": "ActualActualISDA"
}
bond10 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "FixedRateBond",
    "version": 0, 
    "instrumentId": "2400102.IB",
    "start": 2024.08.29,
    "maturity": 2039.08.29,
    "issuePrice": 100,
    "coupon": 0.0225,
    "frequency": "Semiannual",
    "dayCountConvention": "ActualActualISDA"
}
bond11 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "FixedRateBond",
    "version": 0, 
    "instrumentId": "2500004.IB",
    "start": 2025.07.15,
    "maturity": 2045.07.15,
    "issuePrice": 100,
    "coupon": 0.0192,
    "frequency": "Semiannual",
    "dayCountConvention": "ActualActualISDA"
}
bond12 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "FixedRateBond",
    "version": 0, 
    "instrumentId": "2500005.IB",
    "start": 2025.07.15,
    "maturity": 2055.07.15,
    "issuePrice": 100,
    "coupon": 0.019,
    "frequency": "Semiannual",
    "dayCountConvention": "ActualActualISDA"
}
bond13 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "FixedRateBond",
    "version": 0, 
    "instrumentId": "200007.IB",
    "start": 2020.05.25,
    "maturity": 2070.05.25,
    "issuePrice": 100,
    "coupon": 0.0373,
    "frequency": "Semiannual",
    "dayCountConvention": "ActualActualISDA"
}
bond14 = {
    "productType": "Cash",
    "assetType": "Bond",
    "bondType": "FixedRateBond",
    "version": 0, 
    "instrumentId": "2400003.IB",
    "start": 2024.06.15,
    "maturity": 2074.06.15,
    "issuePrice": 100,
    "coupon": 0.0253,
    "frequency": "Semiannual",
    "dayCountConvention": "ActualActualISDA"
}

referenceDate = 2025.08.18
bondsTmp = [bond1, bond2, bond3, bond4, bond5, bond6, bond7, bond8, bond9,
         bond10, bond11, bond12, bond13, bond14]
bonds = parseInstrument(bondsTmp)

```

This example uses the standard tenors from the China Foreign Exchange Trade System \(CFETS, [CFETS Closing Yield Curves- Historical Data - CFETS](https://www.chinamoney.com.cn/english/bmkycvcyccyccychdt/index.html?bondType=CYCC000&reference=1)\), with sample bonds’ remaining terms and quotes being simulated.

``` {#codeblock_mzb_3yq_ngc}
terms = [1M, 3M, 6M, 1y, 2y, 3y, 5y, 7y, 10y, 15y, 20y, 30y, 40y, 50y]
quotes=[1.3000, 1.3700, 1.3898, 1.3865, 1.4296, 1.4466, 1.6348, 
        1.7557, 1.7875, 1.9660, 2.1300, 2.1100, 2.1556, 2.1750]/100
       
// method = "BoostStarp"
bootstrapCurve = bondYieldCurveBuilder(referenceDate, `CNY, bonds, terms, quotes, "ActualActualISDA", method='Bootstrap')
bootstrapCurveDict = extractMktData(bootstrapCurve)
print(bootstrapCurveDict)

// method = "NS"
nsCurve = bondYieldCurveBuilder(referenceDate, `CNY, bonds, terms, quotes, "ActualActualISDA", method='NS')
nsCurveDict = extractMktData(nsCurve)
print(nsCurveDict)

// method = "NSS"
nssCurve = bondYieldCurveBuilder(referenceDate, `CNY, bonds, terms, quotes, "ActualActualISDA", method='NSS')
nssCurveDict=extractMktData(nssCurve)
print(nssCurveDict)

```

**Related functions:** [extractMktData](../e/extractMktData.md), [parseInstrument](../p/parseInstrument.md)

## Bond Product Field Specifications {#topic_vw4_kyq_ngc}

**Discount Bond**

<table id="table_kyz_myq_ngc"><thead><tr><th>

Field Name

</th><th>

Data Type

</th><th>

Description

</th><th>

Required

</th></tr></thead><tbody><tr><td>

productType

</td><td>

STRING

</td><td>

Must be "Cash"

</td><td>

√

</td></tr><tr><td>

assetType

</td><td>

STRING

</td><td>

Must be "Bond"

</td><td>

√

</td></tr><tr><td>

bondType

</td><td>

STRING

</td><td>

Must be "DiscountBond"

</td><td>

√

</td></tr><tr><td>

version

</td><td>

INT

</td><td>

Version, default 0

</td><td>

×

</td></tr><tr><td>

nominal

</td><td>

DOUBLE

</td><td>

Nominal amount, defalut 100

</td><td>

×

</td></tr><tr><td>

instrumentId

</td><td>

STRING

</td><td>

Bond code, e.g., "259926.IB"

</td><td>

×

</td></tr><tr><td>

start

</td><td>

DATE

</td><td>

Value date

</td><td>

√

</td></tr><tr><td>

maturity

</td><td>

DATE

</td><td>

Maturity date

</td><td>

√

</td></tr><tr><td>

dayCountConvention

</td><td>

STRING

</td><td>

The day count convention. It can be: "ActualActualISDA", "ActualActualISMA"," Actual365", "Actual360"

</td><td>

√

</td></tr><tr><td>

issuePrice

</td><td>

DOUBLE

</td><td>

Issue price

</td><td>

√

</td></tr><tr><td>

currency

</td><td>

STRING

</td><td>

Currency, defaults to "CNY"

</td><td>

×

</td></tr><tr><td>

cashFlow

</td><td>

TABLE

</td><td>

Cash flow table

</td><td>

×

</td></tr><tr><td>

discountCurve

</td><td>

STRING

</td><td>

The discount curve, e.g., "CNY\_TRASURY\_BOND"

</td><td>

×

</td></tr><tr><td>

spreadCurve

</td><td>

STRING

</td><td>

The credit spread curve

</td><td>

×

</td></tr><tr><td>

subType

</td><td>

STRING

</td><td>

Subtypes. China's bonds include:

 -   "TREASURY\_BOND": Treasury Bonds

-   "CENTRAL\_BANK\_BILL": Central Bank Bills

-   "CDB\_BOND": Policy Bank Financial Bonds \(China Development Bank\)

-   "EIBC\_BOND": Policy Bank Financial Bonds \(Export-Import Bank of China\)

-   "ADBC\_BOND": Policy Bank Financial Bonds \(Agricultural Development Bank of China\)

-   "MTN": Medium-term Notes

-   "CORP\_BOND": Corporate Bonds

-   "UNSECURED\_CORP\_BOND": Unsecured Corporate Bonds

-   "SHORT\_FIN\_BOND": Short-term Financing Bills

-   "NCD": Negotiable Certificates of Deposit

-   "LOC\_GOV\_BOND": Local Government Bonds

-   "COMM\_BANK\_FIN\_BOND": Commercial Bank Financial Bonds

-   "BANK\_SUB\_CAP\_BOND": Bank Subordinated Capital Bonds

-   "ABS": Asset-backed Securities

-   "PPN": Privately Offered Bonds


</td><td>

×

</td></tr><tr><td>

creditRating

</td><td>

STRING

</td><td>

Credit rating. It can be: "B", "BB", "BBB", "BBB+", "A-", "A", "A+", "AA-", "AA", "AA+", "AAA-", "AAA", "AAA+"

</td><td>

×

</td></tr></tbody>
</table>**Zero Coupon Bond**

<table id="table_nb3_4yq_ngc"><thead><tr><th>

Field Name

</th><th>

Data Type

</th><th>

Description

</th><th>

Required

</th></tr></thead><tbody><tr><td>

productType

</td><td>

STRING

</td><td>

Must be "Cash"

</td><td>

√

</td></tr><tr><td>

assetType

</td><td>

STRING

</td><td>

Must be "Bond"

</td><td>

√

</td></tr><tr><td>

bondType

</td><td>

STRING

</td><td>

Must be "ZeroCouponBond"

</td><td>

√

</td></tr><tr><td>

version

</td><td>

INT

</td><td>

Version, default 0

</td><td>

×

</td></tr><tr><td>

nominal

</td><td>

DOUBLE

</td><td>

Nominal amount, defalut 100

</td><td>

×

</td></tr><tr><td>

instrumentId

</td><td>

STRING

</td><td>

Bond code, e.g., "259926.IB"

</td><td>

×

</td></tr><tr><td>

start

</td><td>

DATE

</td><td>

Value date

</td><td>

√

</td></tr><tr><td>

maturity

</td><td>

DATE

</td><td>

Maturity date

</td><td>

√

</td></tr><tr><td>

coupon

</td><td>

DOUBLE

</td><td>

Coupon rate, e.g., 0.03 means 3%

</td><td>

√

</td></tr><tr><td>

frequency

</td><td>

STRING

</td><td>

Frequency of interest payment

</td><td>

√

</td></tr><tr><td>

dayCountConvention

</td><td>

STRING

</td><td>

The day count convention. It can be: "ActualActualISDA", "ActualActualISMA"," Actual365", "Actual360"

</td><td>

√

</td></tr><tr><td>

currency

</td><td>

STRING

</td><td>

Currency, defaults to "CNY"

</td><td>

×

</td></tr><tr><td>

cashFlow

</td><td>

TABLE

</td><td>

Cash flow table

</td><td>

×

</td></tr><tr><td>

discountCurve

</td><td>

STRING

</td><td>

The discount curve, e.g., "CNY\_TRASURY\_BOND"

</td><td>

×

</td></tr><tr><td>

spreadCurve

</td><td>

STRING

</td><td>

The credit spread curve

</td><td>

×

</td></tr><tr><td>

subType

</td><td>

STRING

</td><td>

Subtypes. China's bonds include:

 -   "TREASURY\_BOND": Treasury Bonds

-   "CENTRAL\_BANK\_BILL": Central Bank Bills

-   "CDB\_BOND": Policy Bank Financial Bonds \(China Development Bank\)

-   "EIBC\_BOND": Policy Bank Financial Bonds \(Export-Import Bank of China\)

-   "ADBC\_BOND": Policy Bank Financial Bonds \(Agricultural Development Bank of China\)

-   "MTN": Medium-term Notes

-   "CORP\_BOND": Corporate Bonds

-   "UNSECURED\_CORP\_BOND": Unsecured Corporate Bonds

-   "SHORT\_FIN\_BOND": Short-term Financing Bills

-   "NCD": Negotiable Certificates of Deposit

-   "LOC\_GOV\_BOND": Local Government Bonds

-   "COMM\_BANK\_FIN\_BOND": Commercial Bank Financial Bonds

-   "BANK\_SUB\_CAP\_BOND": Bank Subordinated Capital Bonds

-   "ABS": Asset-backed Securities

-   "PPN": Privately Offered Bonds


</td><td>

×

</td></tr><tr><td>

creditRating

</td><td>

STRING

</td><td>

Credit rating. It can be: "B", "BB", "BBB", "BBB+", "A-", "A", "A+", "AA-", "AA", "AA+", "AAA-", "AAA", "AAA+"

</td><td>

×

</td></tr></tbody>
</table>**Fixed Rate Bond**

<table id="table_cdt_pyq_ngc"><thead><tr><th>

Field Name

</th><th>

Data Type

</th><th>

Description

</th><th>

Required

</th></tr></thead><tbody><tr><td>

productType

</td><td>

STRING

</td><td>

Must be "Cash"

</td><td>

√

</td></tr><tr><td>

assetType

</td><td>

STRING

</td><td>

Must be "Bond"

</td><td>

√

</td></tr><tr><td>

bondType

</td><td>

STRING

</td><td>

Must be "FixedRateBond"

</td><td>

√

</td></tr><tr><td>

version

</td><td>

INT

</td><td>

Version, default 0

</td><td>

×

</td></tr><tr><td>

nominal

</td><td>

DOUBLE

</td><td>

Nominal amount, defalut 100

</td><td>

×

</td></tr><tr><td>

instrumentId

</td><td>

STRING

</td><td>

Bond code, e.g., "259926.IB"

</td><td>

×

</td></tr><tr><td>

start

</td><td>

DATE

</td><td>

Value date

</td><td>

√

</td></tr><tr><td>

maturity

</td><td>

DATE

</td><td>

Maturity date

</td><td>

√

</td></tr><tr><td>

coupon

</td><td>

DOUBLE

</td><td>

Coupon rate, e.g., 0.03 means 3%

</td><td>

√

</td></tr><tr><td>

frequency

</td><td>

STRING

</td><td>

Frequency of interest payment

</td><td>

√

</td></tr><tr><td>

dayCountConvention

</td><td>

STRING

</td><td>

The day count convention. It can be: "ActualActualISDA", "ActualActualISMA"," Actual365", "Actual360"

</td><td>

√

</td></tr><tr><td>

currency

</td><td>

STRING

</td><td>

Currency, defaults to "CNY"

</td><td>

×

</td></tr><tr><td>

cashFlow

</td><td>

TABLE

</td><td>

Cash flow table

</td><td>

×

</td></tr><tr><td>

discountCurve

</td><td>

STRING

</td><td>

The discount curve, e.g., "CNY\_TRASURY\_BOND"

</td><td>

×

</td></tr><tr><td>

spreadCurve

</td><td>

STRING

</td><td>

The credit spread curve

</td><td>

×

</td></tr><tr><td>

subType

</td><td>

STRING

</td><td>

Subtypes. China's bonds include:

 -   "TREASURY\_BOND": Treasury Bonds

-   "CENTRAL\_BANK\_BILL": Central Bank Bills

-   "CDB\_BOND": Policy Bank Financial Bonds \(China Development Bank\)

-   "EIBC\_BOND": Policy Bank Financial Bonds \(Export-Import Bank of China\)

-   "ADBC\_BOND": Policy Bank Financial Bonds \(Agricultural Development Bank of China\)

-   "MTN": Medium-term Notes

-   "CORP\_BOND": Corporate Bonds

-   "UNSECURED\_CORP\_BOND": Unsecured Corporate Bonds

-   "SHORT\_FIN\_BOND": Short-term Financing Bills

-   "NCD": Negotiable Certificates of Deposit

-   "LOC\_GOV\_BOND": Local Government Bonds

-   "COMM\_BANK\_FIN\_BOND": Commercial Bank Financial Bonds

-   "BANK\_SUB\_CAP\_BOND": Bank Subordinated Capital Bonds

-   "ABS": Asset-backed Securities

-   "PPN": Privately Offered Bonds


</td><td>

×

</td></tr><tr><td>

creditRating

</td><td>

STRING

</td><td>

Credit rating. It can be: "B", "BB", "BBB", "BBB+", "A-", "A", "A+", "AA-", "AA", "AA+", "AAA-", "AAA", "AAA+"

</td><td>

×

</td></tr></tbody>
</table>