import json, time, os, asyncio
import numpy as np
from okx_framework import OKXBaseConfig

class SpotKLineConfig(OKXBaseConfig):

    OKX_WS_URL = "wss://ws.okx.com:8443/ws/v5/business"
    TIMEOUT = 65    
    
    # 1. Modify table name and cache file
    tableName = "Cryptocurrency_minKLineST"
    BUFFER_FILE = "./OKX_KLine_Spot_fail_buffer.jsonl"
    
    # 2. Modify table creation script
    def get_create_table_script(self) -> str:
        return '''
 // Minute-level OHLC data
dbName = "dfs://CryptocurrencyKLine"
tbName = "minKLine"
streamtbName = "Cryptocurrency_minKLineST"
if(!existsDatabase(dbName)){
    //dropDatabase(dbName)
    db = database(dbName,RANGE,2010.01M+(0..20)*12)
}else{ db=database(dbName)}

if(!existsTable(dbName,tbName)){
    colNames = `eventTime`collectionTime`symbolSource`symbol`open`high`low`close`volume`numberOfTrades`quoteVolume`takerBuyBase`takerBuyQuote`volCcy
    colTypes = [TIMESTAMP, TIMESTAMP, SYMBOL, SYMBOL,DOUBLE,DOUBLE,DOUBLE,DOUBLE,DOUBLE,INT,DOUBLE,DOUBLE,DOUBLE,DOUBLE]  
    createPartitionedTable(db,table(1:0,colNames,colTypes),tbName,`eventTime)  
}
colNames = `eventTime`collectionTime`symbolSource`symbol`open`high`low`close`volume`numberOfTrades`quoteVolume`takerBuyBase`takerBuyQuote`volCcy
colTypes = [TIMESTAMP, TIMESTAMP, SYMBOL, SYMBOL,DOUBLE,DOUBLE,DOUBLE,DOUBLE,DOUBLE,INT,DOUBLE,DOUBLE,DOUBLE,DOUBLE]  
enableTableShareAndPersistence(table=keyedStreamTable(`symbolSource`symbol`eventTime, 10000:0, colNames, colTypes), tableName=streamtbName, cacheSize=100000, retentionMinutes=2880)
go
// Write to database via subscription
minKLineTb = loadTable(dbName, tbName)
subscribeTable(tableName=streamtbName, actionName="insertDB", offset=-2, handler=minKLineTb, msgAsTable=true, batchSize=10000, throttle=1, persistOffset=true)
'''

    # 3. Modify subscription parameters
    def get_subscription_args(self, inst_ids: list) -> list:
        return [{"channel": "candle1m", "instId": s} for s in inst_ids]
    
    # 4. Modify message processing
    def handle_message(self, message: str):
        d = json.loads(message)
        if "data" not in d or not d["data"] or int(d["data"][0][8])==0:
            return
        
        j = d["data"][0]

        row = [
            int(j[0])+60000, int(time.time() * 1000),
            "OKX-Spot", ''.join(d["arg"]["instId"].split('-')),
            float(j[1]), float(j[2]), float(j[3]), float(j[4]),
            float(j[5]),None,float(j[7]),None,None,float(j[6])
        ]
        
        try:
            self.realtime_q.put(row, block=False)
        except:
            with self.file_lock, open(self.BUFFER_FILE, "a", encoding="utf-8") as f:
                f.write(json.dumps(row) + "\n")

if __name__ == "__main__":
    # Select the type of data to run
    
    # Contract list
    inst_ids = [
        "BTC-USDT",
        "ETH-USDT",
        "ADA-USDT",
        "ALGO-USDT",
        "BNB-USDT",
        "FIL-USDT",
        "GRT-USDT",
        "LTC-USDT",
        "XRP-USDT"
    ]
    
    # Create configuration based on type
    config = SpotKLineConfig()
    
    # Start data pipeline
    io_thread, okx_thread = config.start(inst_ids)
    
    # Keep the main thread live
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        print(f"\n[{time.strftime('%H:%M:%S')}] Stopping...")
        if okx_thread.ws:
            okx_thread.loop.call_soon_threadsafe(
                asyncio.create_task, okx_thread.ws.stop()
            )
        time.sleep(1.0)
        print(f"[{time.strftime('%H:%M:%S')}] Program exited")
        os._exit(0)