from Binance_framework import BinanceBaseConfig, IOThread, normalize_row
from binance.websocket.um_futures.websocket_client import UMFuturesWebsocketClient
import time, json, numpy as np

class BinanceTradeConfig(BinanceBaseConfig):
    """Binance aggTrade data ingestion config""" 
    tableName = "Cryptocurrency_aggTradeST"
    BUFFER_FILE = "./Binance_aggTrade_fail_buffer.jsonl"
    
    symbols = ["btcusdt","ethusdt","adausdt","algousdt",
               "bnbusdt","fetusdt","grtusdt","ltcusdt","xrpusdt"]
    
    def get_create_table_script(self) -> str:
        return '''
//Tick - aggtrade
dbName = "dfs://CryptocurrencyTick"
tbName = "trade"
streamtbName = "Cryptocurrency_aggTradeST"
colNames = `eventTime`collectionTime`symbolSource`symbol`price`quantity`count`side`tradeId`firstId`lastId
colTypes = [TIMESTAMP, TIMESTAMP, SYMBOL, SYMBOL,DOUBLE,DOUBLE,INT,STRING,LONG,LONG,LONG] 
if(!existsDatabase(dbName)){
    dbDate = database("", VALUE, 2012.01.01..2012.01.30)
    dbSym = database("", HASH, [SYMBOL, 2])
    db = database(dbName, COMPO, [dbDate,dbSym], engine='TSDB')    
}else{db=database(dbName)}
if(!existsTable(dbName,tbName)){ 
    createPartitionedTable(db,table(1:0,colNames,colTypes),tbName,`eventTime`symbol,sortColumns=`symbolSource`symbol`eventTime)  
}
enableTableShareAndPersistence(table=keyedStreamTable(`symbolSource`symbol`lastId, 10000:0, colNames, colTypes), tableName=streamtbName, cacheSize=100000, retentionMinutes=2880)
go

// Subscribe to the stream table and insert data into the DFS table
tradeTb = loadTable(dbName, tbName)
subscribeTable(tableName=streamtbName, actionName="insertDB", offset=-2, handler=tradeTb, msgAsTable=true, batchSize=10000, throttle=1, persistOffset=true)

'''
    
    def create_message_handler(self):
        def message_handler(_, message):
            self.last_received_time = time.time()
            j = json.loads(message)
            if j.get("e") != "aggTrade":
                return

            cols = [
                j['E'] ,                 # eventTime (ms,UTC)
                int(time.time()*1000) ,  # collectionTime
                "Binance-Futures",
                j['s'],float(j["p"]),float(j["q"]),None, "sell" if j["m"] else "buy",
                j['a'],j['f'],j['l']
            ]
            
            try:
                self.realtime_q.put(cols, block=False)
            except:
                # Fallback handling
                with self.file_lock, open(self.BUFFER_FILE, "a", encoding="utf-8") as f:
                    f.write(json.dumps(self.normalize_row(cols), ensure_ascii=False) + "\n")
        
        return message_handler
    
    def start_client_and_subscribe(self):
        client = UMFuturesWebsocketClient(
            on_message=self.create_message_handler(),
            proxies={'http': self.proxy_address, 'https': self.proxy_address}
        )
        
        for s in self.symbols:
            client.agg_trade(symbol=s)
            time.sleep(0.2)
        
        return client


if __name__ == "__main__":
    config = BinanceTradeConfig()
    client = config.start_all()
    
    # Keep the main thread running
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        config.quick_exit()