from Binance_framework import BinanceBaseConfig, IOThread, normalize_row
from binance.websocket.spot.websocket_stream import SpotWebsocketStreamClient
import time, json, numpy as np

class BinanceSpotTickTradeConfig(BinanceBaseConfig):
    tableName = "Cryptocurrency_tickTradeST"
    BUFFER_FILE = "./Binance_STickTrade_fail_buffer.jsonl"
    
    symbols = ["btcusdt"]
    
    def get_create_table_script(self) -> str:
        return '''
//Tick - tickTrade
dbName = "dfs://CryptocurrencyTick"
tbName = "tickTrade"
streamtbName = "Cryptocurrency_tickTradeST"
colNames = `eventTime`collectionTime`symbolSource`symbol`tradeId`price`quantity`quoteQty`side
colTypes = [TIMESTAMP, TIMESTAMP, SYMBOL, SYMBOL,LONG,DOUBLE,DOUBLE,DOUBLE,STRING] 
if(!existsDatabase(dbName)){
    dbDate = database("", VALUE, 2012.01.01..2012.01.30)
    dbSym = database("", HASH, [SYMBOL, 2])
    db = database(dbName, COMPO, [dbDate,dbSym], engine='TSDB')    
}else{db=database(dbName)}
if(!existsTable(dbName,tbName)){ 
    createPartitionedTable(db,table(1:0,colNames,colTypes),tbName,`eventTime`symbol,sortColumns=`symbolSource`symbol`eventTime)  
}
enableTableShareAndPersistence(table=keyedStreamTable(`symbolSource`symbol`tradeId, 10000:0, colNames, colTypes), tableName=streamtbName, cacheSize=100000, retentionMinutes=2880)
go

// Subscribe to the stream table and insert data into the DFS table
tradeTb = loadTable(dbName, tbName)
subscribeTable(tableName=streamtbName, actionName="insertDB", offset=-2, handler=tradeTb, msgAsTable=true, batchSize=10000, throttle=1, persistOffset=true)
'''

    def create_message_handler(self):
        def message_handler(_, message):
            self.last_received_time = time.time()
            j = json.loads(message)
            if j.get("e") != "trade":
                return

            cols = [
                j['E'],                # event_time
                int(time.time()*1000), # collection_time
                "Binance-Spot",         # exchange
                j['s'],                 # symbol
                j["t"],
                float(j["p"]),
                float(j["q"]),
                None,
                "sell" if j["m"] else "buy"
            ]
            try:
                self.realtime_q.put(cols, block=False)
            except:
                # Fallback handling
                with self.file_lock, open(self.BUFFER_FILE, "a", encoding="utf-8") as f:
                    f.write(json.dumps(self.normalize_row(cols), ensure_ascii=False) + "\n")
        
        return message_handler
    
    def start_client_and_subscribe(self):
        client = SpotWebsocketStreamClient(
            on_message=self.create_message_handler(),
            proxies={'http': self.proxy_address, 'https': self.proxy_address}
        )
        
        for s in self.symbols:
            client.trade(symbol=s)
            time.sleep(0.2)
        
        return client


if __name__ == "__main__":
    config = BinanceSpotTickTradeConfig()
    client = config.start_all()
    
    # Keep the main thread running
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        config.quick_exit()