from Binance_framework import BinanceBaseConfig, IOThread, normalize_row
from binance.websocket.um_futures.websocket_client import UMFuturesWebsocketClient
import time, json, numpy as np

class BinanceContinuousKLineConfig(BinanceBaseConfig):

    """Binance ContinuousKLine data ingestion config"""
    tableName = "Cryptocurrency_ContinuousKLineST"
    BUFFER_FILE = "./Binance_ContinuousKLine_fail_buffer.jsonl"
    
    symbols = ["btcusdt","ethusdt","adausdt","algousdt",
               "bnbusdt","fetusdt","grtusdt","ltcusdt","xrpusdt"]
    
    def get_create_table_script(self) -> str:
        return '''
dbName = "dfs://CryptocurrencyKLine"
tbName = "ContinuousKLine"
if(!existsDatabase(dbName)){
    //dropDatabase(dbName)
    db = database(dbName,RANGE,2010.01M+(0..20)*12)
}else{ db=database(dbName)}

if(!existsTable(dbName,tbName)){
    colNames = `eventTime`collectionTime`symbolSource`symbol`contractType`open`high`low`close`volume`numberOfTrades`quoteVolume`takerBuyBase`takerBuyQuote`volCcy
    colTypes = [TIMESTAMP, TIMESTAMP, SYMBOL, SYMBOL,STRING, DOUBLE,DOUBLE,DOUBLE,DOUBLE,DOUBLE,INT,DOUBLE,DOUBLE,DOUBLE,DOUBLE]  
    createPartitionedTable(db,table(1:0,colNames,colTypes),tbName,`eventTime)  
}
colNames = `eventTime`collectionTime`symbolSource`symbol`contractType`open`high`low`close`volume`numberOfTrades`quoteVolume`takerBuyBase`takerBuyQuote`volCcy
colTypes = [TIMESTAMP, TIMESTAMP, SYMBOL, SYMBOL,STRING, DOUBLE,DOUBLE,DOUBLE,DOUBLE,DOUBLE,INT,DOUBLE,DOUBLE,DOUBLE,DOUBLE]  
enableTableShareAndPersistence(table=keyedStreamTable(`symbolSource`symbol`eventTime`contractType, 10000:0, colNames, colTypes), tableName="Cryptocurrency_ContinuousKLineST", cacheSize=100000, retentionMinutes=2880)
go

// Subscribe to the stream table and insert data into the DFS table
minKLineTb = loadTable(dbName, tbName)
subscribeTable(tableName="Cryptocurrency_ContinuousKLineST", actionName="insertDB", offset=-2, handler=minKLineTb, msgAsTable=true, batchSize=10000, throttle=1, persistOffset=true)
        '''
    
    def create_message_handler(self):
        def message_handler(_, message):
            self.last_received_time = time.time()
            j = json.loads(message)
            if j.get("e") != "continuous_kline":
                return

            k = j["k"]
            if not k["x"]:
                return 

            cols = [
                j['E'],                # event_time
                int(time.time()*1000), # collection_time
                "Binance-Futures",         # exchange
                j['ps'],                 # symbol
                j["ct"],                 # contractType
                float(k["o"]),
                float(k["h"]),
                float(k["l"]),
                float(k["c"]),
                float(k["v"]),
                k["n"],
                float(k["q"]),
                float(k["V"]),    
                float(k["Q"]),
                None
            ]
            try:
                self.realtime_q.put(cols, block=False)
            except:
                # Fallback handling
                with self.file_lock, open(self.BUFFER_FILE, "a", encoding="utf-8") as f:
                    f.write(json.dumps(self.normalize_row(cols), ensure_ascii=False) + "\n")
        
        return message_handler
    
    def start_client_and_subscribe(self):
        client = UMFuturesWebsocketClient(
            on_message=self.create_message_handler(),
            proxies={'http': self.proxy_address, 'https': self.proxy_address}
        )
        
        for s in self.symbols:
            client.continuous_kline(pair=s, contractType="perpetual", interval="1m")
            time.sleep(0.2)
        
        return client


if __name__ == "__main__":
    config = BinanceContinuousKLineConfig()
    client = config.start_all()
    
    # Keep the main thread running
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        config.quick_exit()